import json
import importlib
import glob
import sys
import os
import logging


logger = logging.getLogger("it-sec.checker")


with open(os.path.join(os.path.abspath(os.path.dirname(__file__)), "settings.json"), "r") as f:
    settings = json.load(f)

    logger.info("done loading settings")

# load all dynamic modules
imports = {}

if "imports" in settings:
    sys.path.insert(1, "")
    tmp = os.path.abspath(os.path.dirname(__file__))

    for i in settings["imports"]:
        sys.path[1] = os.path.join(tmp, i)

        for j in glob.glob(os.path.join(tmp, i, "*.py")):
            name = os.path.basename(os.path.splitext(j)[0])
            imports[name] = importlib.import_module(name)

            logger.info("added import %s", name)

    del sys.path[1]


def _check(item, ref):
    def create(status, **kwargs):
        tmp = {
            "name": item["name"],
            "ref": ref,
            "status": "success" if status else "error",
            "priority": item["priority"]
        }

        for key, value in kwargs.items():
            tmp[key] = value

        if "description" in item:
            tmp["description"] = item["description"]

        return tmp

    if item["type"] == "category":
        children = []

        for i, v in enumerate(item["children"]):
            c = _check(v, "{}.{}".format(ref, i+1))

            if c is not None:
                children.append(c)

        return None if len(children) == 0 else create(all(i["status"] == "success" for i in children), children=children, fixable=any(i["fixable"] and i["status"] != "success" for i in children))

    try:
        privileged = item["privileged"]
    except:
        privileged = "super-fix"

    if privileged == "super" and os.getuid() != 0:
        return

    try:
        rc = eval(item["check"], imports, None)
    except:
        logger.exception("failed to check rule: %s", item["name"])

        rc = False

    return create(rc, fixable="fix" in item and (os.getuid() == 0 or privileged == "no-super"))


def _get_by_ref(ref, items):
    if len(ref) == 1:
        return items[ref[0]]
    return _get_by_ref(ref[1:], items[ref[0]]["children"])


def check():
    for i, v in enumerate(settings["settings"]):
        c = _check(v, str(i+1))

        if c is not None:
            yield c


def fix(refs):
    for ref in refs:
        logger.debug("fixing rule: %s", ref)

        item = _get_by_ref(
            [int(i) - 1 for i in ref.split(".")], settings["settings"])
        error = None

        try:
            if not eval(item["fix"], imports, None):
                logger.error("failed to fix rule: %s", item["name"])

                error = "Something went wrong with {}".format(ref)
        except Exception as e:
            logger.exception("failed to fix rule: %s", item["name"])

            error = "error for rule ({}) {}".format(item["name"], e)

        # log rc
        yield error
