import React from 'react';
import { withStyles, Paper, Backdrop, CircularProgress, Typography } from "@material-ui/core";
import QuickActions from "./QuickActions";
import RuleDisplay from "./RuleDisplay";
import printer from "./printer";
import { withSnackbar } from "notistack";
import ConfirmDialog from "./ConfirmDialog";

const styles = theme => ({
	root: {
		display: "flex",
		minHeight: "100vh",
		maxHeight: "100vh",
		overflow: "hidden",
		backgroundColor: theme.palette.background.default
	},
	paper: {
		margin: theme.spacing(2),
		overflowX: "auto",
		overflowY: "scroll",
		flexGrow: 1
	},
	backdrop: {
		zIndex: theme.zIndex.drawer + 1
	},
	progressNumber: {
		position: "absolute"
	}
});

class App extends React.Component {
	constructor(props) {
		super(props);

		this.state = {
			settings: {
				rules: []
			},
			warningCounts: {},
			selected: new Set(),
			expanded: new Set(),
			loading: true,
			progress: null,
			confirm: false
		};
		this._fetchSettings = () => fetch("/api/settings")
			.then(x => x.json())
			.then(x => {
				const warningCounts = { low: 0, medium: 0, high: 0 };
				const rec = y => {
					if (y.hasOwnProperty("children")) {
						for (let i of y.children) {
							rec(i);
						}
					} else {
						warningCounts[y.priority] += y.status === "success" ? 0 : 1;
					}
				};

				for (let i of x.rules) {
					rec(i);
				}

				this.setState({ ...this.state, settings: x, warningCounts: warningCounts, loading: false })
			});
	}

	componentDidMount() {
		this._fetchSettings();
	}

	render() {
		const { classes } = this.props;
		const selectAllIssues = () => {
			const selected = new Set();
			const select = x => {
				if (x.hasOwnProperty("children")) {
					for (let i of x.children) {
						select(i);
					}
				} else if (x.status !== "success" && x.fixable) {
					selected.add(x.ref);
				}
			};

			for (let i of this.state.settings.rules) {
				select(i);
			}

			this.setState({ ...this.state, selected: selected });
		};
		const onExpandToggle = x => {
			const tmp = new Set([...this.state.expanded]);

			if (!tmp.delete(x.ref)) {
				tmp.add(x.ref);
			}

			this.setState({ ...this.state, expanded: tmp });
		};
		const onExpandAll = () => {
			const tmp = new Set();

			const rec = item => {
				if (item.hasOwnProperty("children")) {
					tmp.add(item.ref);

					for (let i of item.children) {
						rec(i);
					}
				}
			};

			for (let i of this.state.settings.rules) {
				rec(i);
			}

			this.setState({ ...this.state, expanded: tmp });
		};
		const fixIssues = () => {
			const issues = [];
			const byRef = ref => {
				const indices = ref.split(".");
				let rules = { children: this.state.settings.rules };

				for (let i of indices) {
					rules = rules.children[i - 1];
				}

				return rules;
			};

			for (let i of this.state.selected) {
				if (!byRef(i).hasOwnProperty("children")) {
					issues.push(i);
				}
			}

			const readChunk = reader => reader.read().then(chunk => {
				if (chunk.done) {
					this.props.enqueueSnackbar("The fixes have been applied.", { variant: "success" });
					this.setState({ ...this.state, loading: false, selected: new Set() });
					this._fetchSettings();
				} else {
					var str = new TextDecoder("utf-8").decode(chunk.value);
					var msg;

					while (str) {
						const index = str.indexOf(";");

						msg = JSON.parse(str.substr(0, index));

						if (msg.error) {
							console.error(msg.error);
							this.props.enqueueSnackbar(msg.error, { variant: "error" });
						}

						str = str.substr(index + 1);
					}

					this.setState({
						...this.state, progress: msg.progress
					});

					return readChunk(reader);
				}
			});

			this.setState({ ...this.state, loading: true, confirm: false });

			if (issues.length) {
				fetch("/api/fix", {
					method: "POST",
					headers: {
						"Content-Type": "application/json"
					},
					body: JSON.stringify(issues)
				})
					.then(resp => {
						if (!resp.ok) {
							throw resp;
						}

						return readChunk(resp.body.getReader());
					})
					.catch(console.error);
			}
		};

		return (
			<div className={classes.root}>
				<Backdrop
					open={this.state.loading}
					className={classes.backdrop}>
					<CircularProgress color="secondary" size={60} />

					{
						this.state.progress == null ? null :
							<Typography className={classes.progressNumber} variant="body1" color="textPrimary">{this.state.progress} %</Typography>
					}
				</Backdrop>

				<QuickActions
					hostname={this.state.settings.hostname}
					warningCounts={this.state.warningCounts}
					onExpandAll={onExpandAll}
					onFix={() => this.setState({ ...this.state, confirm: true })}
					onSave={() => printer(this.state.settings.rules)}
					onCollapseAll={() => this.setState({ ...this.state, expanded: new Set() })}
					onClearSelection={() => this.setState({ ...this.state, selected: new Set() })}
					onSelectIssues={selectAllIssues} />

				<ConfirmDialog
					open={this.state.confirm}
					onCancel={() => this.setState({ ...this.state, confirm: false })}
					onConfirm={fixIssues} >
					<Typography variant="body1">Apply all selected fixes to your machine?</Typography>
				</ConfirmDialog>

				<Paper className={classes.paper}>
					<RuleDisplay
						expanded={this.state.expanded}
						onExpandToggle={onExpandToggle}
						selected={this.state.selected}
						setSelected={x => this.setState({ ...this.state, selected: x })}
						rules={this.state.settings.rules} />
				</Paper>
			</div>
		);
	}
}

export default withStyles(styles)(withSnackbar(App));
